<?php
/**
 * Installation API
 * API สำหรับการติดตั้งระบบ
 */

require_once '../../config/config.php';

// ตั้งค่า Content-Type
header('Content-Type: application/json; charset=utf-8');

// ตรวจสอบ HTTP Method
$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

try {
    switch ($action) {
        case 'check':
            handleSystemCheck();
            break;
        case 'test-db':
            handleDatabaseTest();
            break;
        case 'create-db':
            handleCreateDatabase();
            break;
        case 'create-admin':
            handleCreateAdmin();
            break;
        default:
            sendJsonResponse(['error' => 'Action not found'], 404);
    }
    
} catch (Exception $e) {
    error_log("Install API error: " . $e->getMessage());
    sendJsonResponse(['error' => 'เกิดข้อผิดพลาดในระบบ'], 500);
}

// ตรวจสอบระบบ
function handleSystemCheck() {
    $checks = [];
    $allPassed = true;
    
    // ตรวจสอบ PHP Version
    $phpVersion = PHP_VERSION;
    $phpRequired = '7.4.0';
    $phpCheck = version_compare($phpVersion, $phpRequired, '>=');
    $checks['php'] = [
        'name' => 'PHP Version',
        'required' => $phpRequired,
        'current' => $phpVersion,
        'passed' => $phpCheck
    ];
    if (!$phpCheck) $allPassed = false;
    
    // ตรวจสอบ Extensions
    $requiredExtensions = ['pdo', 'pdo_mysql', 'json', 'mbstring', 'openssl'];
    foreach ($requiredExtensions as $ext) {
        $extCheck = extension_loaded($ext);
        $checks['ext_' . $ext] = [
            'name' => 'PHP Extension: ' . $ext,
            'required' => 'Installed',
            'current' => $extCheck ? 'Installed' : 'Not Installed',
            'passed' => $extCheck
        ];
        if (!$extCheck) $allPassed = false;
    }
    
    // ตรวจสอบ Permissions
    $writableDirs = ['uploads', 'config'];
    foreach ($writableDirs as $dir) {
        $dirPath = '../../' . $dir;
        $writable = is_dir($dirPath) && is_writable($dirPath);
        $checks['dir_' . $dir] = [
            'name' => 'Directory: ' . $dir,
            'required' => 'Writable',
            'current' => $writable ? 'Writable' : 'Not Writable',
            'passed' => $writable
        ];
        if (!$writable) $allPassed = false;
    }
    
    // ตรวจสอบ Memory Limit
    $memoryLimit = ini_get('memory_limit');
    $memoryBytes = return_bytes($memoryLimit);
    $memoryCheck = $memoryBytes >= 128 * 1024 * 1024; // 128MB
    $checks['memory'] = [
        'name' => 'Memory Limit',
        'required' => '128M',
        'current' => $memoryLimit,
        'passed' => $memoryCheck
    ];
    if (!$memoryCheck) $allPassed = false;
    
    // ตรวจสอบ Upload Settings
    $uploadMaxFilesize = ini_get('upload_max_filesize');
    $postMaxSize = ini_get('post_max_size');
    $uploadCheck = return_bytes($uploadMaxFilesize) >= 5 * 1024 * 1024; // 5MB
    $checks['upload'] = [
        'name' => 'Upload Settings',
        'required' => '5M',
        'current' => $uploadMaxFilesize,
        'passed' => $uploadCheck
    ];
    if (!$uploadCheck) $allPassed = false;
    
    if ($allPassed) {
        sendJsonResponse([
            'success' => true,
            'message' => 'ระบบพร้อมสำหรับการติดตั้ง',
            'checks' => $checks
        ]);
    } else {
        sendJsonResponse([
            'success' => false,
            'message' => 'ระบบไม่พร้อมสำหรับการติดตั้ง',
            'checks' => $checks
        ]);
    }
}

// ทดสอบการเชื่อมต่อฐานข้อมูล
function handleDatabaseTest() {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        sendJsonResponse(['error' => 'Method not allowed'], 405);
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || empty($input['dbHost']) || empty($input['dbName']) || empty($input['dbUser'])) {
        sendJsonResponse(['error' => 'ข้อมูลไม่ครบถ้วน'], 400);
    }
    
    try {
        $dsn = "mysql:host={$input['dbHost']};charset=utf8mb4";
        $pdo = new PDO($dsn, $input['dbUser'], $input['dbPass'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
        ]);
        
        // ตรวจสอบว่าฐานข้อมูลมีอยู่หรือไม่
        $stmt = $pdo->prepare("SHOW DATABASES LIKE ?");
        $stmt->execute([$input['dbName']]);
        $dbExists = $stmt->fetch() !== false;
        
        sendJsonResponse([
            'success' => true,
            'message' => 'เชื่อมต่อฐานข้อมูลสำเร็จ',
            'database_exists' => $dbExists
        ]);
        
    } catch (PDOException $e) {
        sendJsonResponse([
            'success' => false,
            'message' => 'ไม่สามารถเชื่อมต่อฐานข้อมูลได้: ' . $e->getMessage()
        ]);
    }
}

// สร้างฐานข้อมูล
function handleCreateDatabase() {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        sendJsonResponse(['error' => 'Method not allowed'], 405);
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || empty($input['dbHost']) || empty($input['dbName']) || empty($input['dbUser'])) {
        sendJsonResponse(['error' => 'ข้อมูลไม่ครบถ้วน'], 400);
    }
    
    try {
        // เชื่อมต่อ MySQL Server
        $dsn = "mysql:host={$input['dbHost']};charset=utf8mb4";
        $pdo = new PDO($dsn, $input['dbUser'], $input['dbPass'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
        ]);
        
        // สร้างฐานข้อมูล
        $pdo->exec("CREATE DATABASE IF NOT EXISTS `{$input['dbName']}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        
        // เลือกฐานข้อมูล
        $pdo->exec("USE `{$input['dbName']}`");
        
        // อ่านและรัน SQL Schema
        $schemaFile = '../../database/schema.sql';
        if (!file_exists($schemaFile)) {
            throw new Exception('ไม่พบไฟล์ schema.sql');
        }
        
        $sql = file_get_contents($schemaFile);
        
        // แยกคำสั่ง SQL
        $statements = explode(';', $sql);
        foreach ($statements as $statement) {
            $statement = trim($statement);
            if (!empty($statement)) {
                $pdo->exec($statement);
            }
        }
        
        // สร้างไฟล์ config
        createConfigFile($input);
        
        sendJsonResponse([
            'success' => true,
            'message' => 'สร้างฐานข้อมูลและตารางสำเร็จ'
        ]);
        
    } catch (Exception $e) {
        error_log("Create database error: " . $e->getMessage());
        sendJsonResponse([
            'success' => false,
            'message' => 'ไม่สามารถสร้างฐานข้อมูลได้: ' . $e->getMessage()
        ]);
    }
}

// สร้างบัญชีผู้ดูแลระบบ
function handleCreateAdmin() {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        sendJsonResponse(['error' => 'Method not allowed'], 405);
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || empty($input['adminUsername']) || empty($input['adminPassword']) || 
        empty($input['adminEmail']) || empty($input['adminName'])) {
        sendJsonResponse(['error' => 'ข้อมูลไม่ครบถ้วน'], 400);
    }
    
    try {
        // เชื่อมต่อฐานข้อมูล
        $dsn = "mysql:host={$input['dbHost']};dbname={$input['dbName']};charset=utf8mb4";
        $pdo = new PDO($dsn, $input['dbUser'], $input['dbPass'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
        ]);
        
        // ตรวจสอบว่ามีผู้ใช้อยู่แล้วหรือไม่
        $stmt = $pdo->prepare("SELECT id FROM users WHERE username = ? OR email = ?");
        $stmt->execute([$input['adminUsername'], $input['adminEmail']]);
        
        if ($stmt->fetch()) {
            sendJsonResponse([
                'success' => false,
                'message' => 'ชื่อผู้ใช้หรืออีเมลนี้มีอยู่แล้ว'
            ]);
        }
        
        // สร้างผู้ใช้ใหม่
        $hashedPassword = password_hash($input['adminPassword'], PASSWORD_DEFAULT);
        
        $stmt = $pdo->prepare("
            INSERT INTO users (username, email, password, full_name, role, is_active) 
            VALUES (?, ?, ?, ?, 'super_admin', 1)
        ");
        
        $stmt->execute([
            $input['adminUsername'],
            $input['adminEmail'],
            $hashedPassword,
            $input['adminName']
        ]);
        
        // สร้างไฟล์ .htaccess เพื่อป้องกันการเข้าถึง install.php
        createHtaccessFile();
        
        sendJsonResponse([
            'success' => true,
            'message' => 'สร้างบัญชีผู้ดูแลระบบสำเร็จ'
        ]);
        
    } catch (Exception $e) {
        error_log("Create admin error: " . $e->getMessage());
        sendJsonResponse([
            'success' => false,
            'message' => 'ไม่สามารถสร้างบัญชีผู้ดูแลได้: ' . $e->getMessage()
        ]);
    }
}

// สร้างไฟล์ config
function createConfigFile($dbConfig) {
    $configContent = "<?php
/**
 * Database Configuration
 * Generated by installer
 */

// Database settings
define('DB_HOST', '{$dbConfig['dbHost']}');
define('DB_NAME', '{$dbConfig['dbName']}');
define('DB_USER', '{$dbConfig['dbUser']}');
define('DB_PASS', '{$dbConfig['dbPass']}');

// Application settings
define('APP_URL', 'https://hr.maesai.ac.th');
define('ADMIN_EMAIL', '{$dbConfig['adminEmail']}');

// Security settings
define('JWT_SECRET', '" . bin2hex(random_bytes(32)) . "');
define('PASSWORD_MIN_LENGTH', 8);
define('SESSION_TIMEOUT', 3600);

// Upload settings
define('UPLOAD_PATH', 'uploads/');
define('MAX_FILE_SIZE', 5 * 1024 * 1024);

// Other settings
define('ITEMS_PER_PAGE', 20);
define('API_VERSION', 'v1');
define('API_BASE_URL', APP_URL . '/api/' . API_VERSION);
?>";
    
    file_put_contents('../../config/config.local.php', $configContent);
}

// สร้างไฟล์ .htaccess
function createHtaccessFile() {
    $htaccessContent = "# Prevent access to install files
<Files \"install.php\">
    Order Allow,Deny
    Deny from all
</Files>

<Files \"api/v1/install.php\">
    Order Allow,Deny
    Deny from all
</Files>";
    
    file_put_contents('../../.htaccess', $htaccessContent, FILE_APPEND);
}

// ฟังก์ชันช่วยเหลือ
function return_bytes($val) {
    $val = trim($val);
    $last = strtolower($val[strlen($val)-1]);
    $val = (int) $val;
    
    switch($last) {
        case 'g':
            $val *= 1024;
        case 'm':
            $val *= 1024;
        case 'k':
            $val *= 1024;
    }
    
    return $val;
}
?>
