<?php
/**
 * ระบบบริหารงานบุคคล โรงเรียนแม่สายประสิทธิ์ศาสตร์
 * Main Configuration File
 */

// ตั้งค่า Error Reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// ตั้งค่า Timezone
date_default_timezone_set('Asia/Bangkok');

// ตั้งค่า Session
ini_set('session.cookie_httponly', 1);
ini_set('session.use_only_cookies', 1);
ini_set('session.cookie_secure', 0); // เปลี่ยนเป็น 1 เมื่อใช้ HTTPS
ini_set('session.cookie_samesite', 'Strict');
ini_set('session.use_strict_mode', 1);

// เริ่ม Session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// ตั้งค่าฐานข้อมูล
define('DB_HOST', $_ENV['DB_HOST'] ?? 'localhost');
define('DB_NAME', $_ENV['DB_NAME'] ?? 'hr_maesai');
define('DB_USER', $_ENV['DB_USER'] ?? 'root');
define('DB_PASS', $_ENV['DB_PASS'] ?? '');

// ตั้งค่าแอปพลิเคชัน
define('APP_NAME', 'ระบบบริหารงานบุคคล');
define('APP_VERSION', '1.0.0');
define('APP_URL', $_ENV['APP_URL'] ?? 'http://localhost');
define('ADMIN_EMAIL', 'admin@maesai.ac.th');

// ตั้งค่าการอัปโหลดไฟล์
define('UPLOAD_PATH', 'uploads/');
define('MAX_FILE_SIZE', 5 * 1024 * 1024); // 5MB
define('ALLOWED_IMAGE_TYPES', ['jpg', 'jpeg', 'png', 'gif']);

// ตั้งค่าการส่งอีเมล
define('SMTP_HOST', $_ENV['SMTP_HOST'] ?? 'localhost');
define('SMTP_PORT', $_ENV['SMTP_PORT'] ?? 587);
define('SMTP_USERNAME', $_ENV['SMTP_USERNAME'] ?? '');
define('SMTP_PASSWORD', $_ENV['SMTP_PASSWORD'] ?? '');

// ตั้งค่าความปลอดภัย
define('JWT_SECRET', $_ENV['JWT_SECRET'] ?? 'your-secret-key-change-this');
define('PASSWORD_MIN_LENGTH', 8);
define('SESSION_TIMEOUT', 3600); // 1 hour

// ตั้งค่า Pagination
define('ITEMS_PER_PAGE', 20);

// ตั้งค่า API
define('API_VERSION', 'v1');
define('API_BASE_URL', APP_URL . '/api/' . API_VERSION);

// ฟังก์ชันช่วยเหลือ
function sanitizeInput($data) {
    return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
}

function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

function generateToken($length = 32) {
    return bin2hex(random_bytes($length));
}

function hashPassword($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

function formatDate($date, $format = 'd/m/Y') {
    return date($format, strtotime($date));
}

function formatThaiDate($date) {
    $thai_months = [
        1 => 'มกราคม', 2 => 'กุมภาพันธ์', 3 => 'มีนาคม', 4 => 'เมษายน',
        5 => 'พฤษภาคม', 6 => 'มิถุนายน', 7 => 'กรกฎาคม', 8 => 'สิงหาคม',
        9 => 'กันยายน', 10 => 'ตุลาคม', 11 => 'พฤศจิกายน', 12 => 'ธันวาคม'
    ];
    
    $timestamp = strtotime($date);
    $day = date('j', $timestamp);
    $month = $thai_months[(int)date('n', $timestamp)];
    $year = date('Y', $timestamp) + 543; // แปลงเป็น พ.ศ.
    
    return "$day $month $year";
}

function sendJsonResponse($data, $status_code = 200) {
    http_response_code($status_code);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

function logActivity($user_id, $action, $details = '') {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO audit_logs (user_id, table_name, record_id, action, new_values, ip_address) 
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $user_id,
            'system',
            0,
            $action,
            json_encode(['details' => $details]),
            $_SERVER['REMOTE_ADDR'] ?? 'unknown'
        ]);
    } catch (Exception $e) {
        error_log("Log activity error: " . $e->getMessage());
    }
}

// ตรวจสอบการเข้าถึง API
function checkApiAccess() {
    $headers = getallheaders();
    $auth_header = $headers['Authorization'] ?? '';
    
    if (empty($auth_header) || !preg_match('/Bearer\s+(.*)$/i', $auth_header, $matches)) {
        sendJsonResponse(['error' => 'ไม่ได้รับอนุญาตให้เข้าถึง'], 401);
    }
    
    $token = $matches[1];
    // TODO: ตรวจสอบ JWT token
    return true;
}

// ตรวจสอบสิทธิ์ผู้ดูแลระบบ
function requireAdmin() {
    if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin' && $_SESSION['role'] !== 'super_admin') {
        sendJsonResponse(['error' => 'ต้องเป็นผู้ดูแลระบบเท่านั้น'], 403);
    }
}

// ตรวจสอบการเข้าสู่ระบบ
function requireLogin() {
    if (!isset($_SESSION['user_id'])) {
        sendJsonResponse(['error' => 'ต้องเข้าสู่ระบบก่อน'], 401);
    }
}

// ตั้งค่า CORS (เฉพาะ API)
function setCorsHeaders() {
    // ตรวจสอบว่าเป็น API request หรือไม่
    if (strpos($_SERVER['REQUEST_URI'], '/api/') !== false) {
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
        header('Access-Control-Allow-Headers: Content-Type, Authorization');
        
        if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
            http_response_code(200);
            exit;
        }
    }
}

// เรียกใช้ CORS
setCorsHeaders();

// Auto-load classes
spl_autoload_register(function ($class_name) {
    $directories = [
        'classes/',
        'models/',
        'controllers/',
        'middleware/'
    ];
    
    foreach ($directories as $directory) {
        $file = $directory . $class_name . '.php';
        if (file_exists($file)) {
            require_once $file;
            return;
        }
    }
});

// ตั้งค่า Error Handler
set_error_handler(function($severity, $message, $file, $line) {
    if (!(error_reporting() & $severity)) {
        return false;
    }
    
    $error_data = [
        'severity' => $severity,
        'message' => $message,
        'file' => $file,
        'line' => $line,
        'timestamp' => date('Y-m-d H:i:s')
    ];
    
    error_log(json_encode($error_data));
    
    if ($severity === E_ERROR || $severity === E_USER_ERROR) {
        sendJsonResponse(['error' => 'เกิดข้อผิดพลาดในระบบ'], 500);
    }
    
    return true;
});

// ตั้งค่า Exception Handler
set_exception_handler(function($exception) {
    error_log("Uncaught exception: " . $exception->getMessage());
    sendJsonResponse(['error' => 'เกิดข้อผิดพลาดในระบบ'], 500);
});

?>
