<?php
/**
 * Personnel API
 * API สำหรับจัดการข้อมูลบุคลากร
 */

require_once '../../config/config.php';
require_once '../../models/Personnel.php';

// ตั้งค่า Content-Type
header('Content-Type: application/json; charset=utf-8');

// ตรวจสอบ HTTP Method
$method = $_SERVER['REQUEST_METHOD'];

try {
    $personnel = new Personnel();
    
    switch ($method) {
        case 'GET':
            handleGet($personnel);
            break;
        case 'POST':
            handlePost($personnel);
            break;
        case 'PUT':
            handlePut($personnel);
            break;
        case 'DELETE':
            handleDelete($personnel);
            break;
        default:
            sendJsonResponse(['error' => 'Method not allowed'], 405);
    }
    
} catch (Exception $e) {
    error_log("Personnel API error: " . $e->getMessage());
    sendJsonResponse(['error' => 'เกิดข้อผิดพลาดในระบบ'], 500);
}

// GET - ดึงข้อมูลบุคลากร
function handleGet($personnel) {
    $id = isset($_GET['id']) ? (int)$_GET['id'] : null;
    $action = $_GET['action'] ?? 'list';
    
    if ($action === 'stats') {
        // ดึงสถิติบุคลากร
        $stats = $personnel->getStats();
        if ($stats) {
            sendJsonResponse([
                'success' => true,
                'data' => $stats
            ]);
        } else {
            sendJsonResponse(['error' => 'ไม่สามารถดึงข้อมูลสถิติได้'], 500);
        }
    } elseif ($id) {
        // ดึงข้อมูลบุคลากรตาม ID
        $data = $personnel->getById($id);
        if ($data) {
            sendJsonResponse([
                'success' => true,
                'data' => $data
            ]);
        } else {
            sendJsonResponse(['error' => 'ไม่พบข้อมูลบุคลากร'], 404);
        }
    } else {
        // ดึงรายการบุคลากร
        $search = sanitizeInput($_GET['search'] ?? '');
        $rank_filter = sanitizeInput($_GET['rank'] ?? '');
        $gender_filter = sanitizeInput($_GET['gender'] ?? '');
        $page = max(1, (int)($_GET['page'] ?? 1));
        $limit = min(100, max(1, (int)($_GET['limit'] ?? 20)));
        $offset = ($page - 1) * $limit;
        
        $data = $personnel->getAll($limit, $offset, $search, $rank_filter, $gender_filter);
        $total = $personnel->count($search, $rank_filter, $gender_filter);
        
        if ($data !== false) {
            sendJsonResponse([
                'success' => true,
                'data' => $data,
                'pagination' => [
                    'current_page' => $page,
                    'total_pages' => ceil($total / $limit),
                    'total_items' => $total,
                    'items_per_page' => $limit
                ]
            ]);
        } else {
            sendJsonResponse(['error' => 'ไม่สามารถดึงข้อมูลบุคลากรได้'], 500);
        }
    }
}

// POST - สร้างบุคลากรใหม่
function handlePost($personnel) {
    // ตรวจสอบสิทธิ์
    requireAdmin();
    
    // รับข้อมูลจาก Request Body
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        sendJsonResponse(['error' => 'ข้อมูลไม่ถูกต้อง'], 400);
    }
    
    // ตรวจสอบข้อมูลที่จำเป็น
    $required_fields = ['personnel_id', 'first_name', 'last_name'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            sendJsonResponse(['error' => "ข้อมูล $field จำเป็นต้องกรอก"], 400);
        }
    }
    
    // สร้างบุคลากร
    $personnel_id = $personnel->create($input);
    
    if ($personnel_id) {
        sendJsonResponse([
            'success' => true,
            'message' => 'สร้างข้อมูลบุคลากรสำเร็จ',
            'data' => ['id' => $personnel_id]
        ], 201);
    } else {
        sendJsonResponse(['error' => 'ไม่สามารถสร้างข้อมูลบุคลากรได้'], 500);
    }
}

// PUT - อัปเดตข้อมูลบุคลากร
function handlePut($personnel) {
    // ตรวจสอบสิทธิ์
    requireAdmin();
    
    $id = isset($_GET['id']) ? (int)$_GET['id'] : null;
    if (!$id || $id <= 0) {
        sendJsonResponse(['error' => 'ไม่ระบุ ID บุคลากร'], 400);
    }
    
    // รับข้อมูลจาก Request Body
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        sendJsonResponse(['error' => 'ข้อมูลไม่ถูกต้อง'], 400);
    }
    
    // ตรวจสอบข้อมูลที่จำเป็น
    $required_fields = ['first_name', 'last_name'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            sendJsonResponse(['error' => "ข้อมูล $field จำเป็นต้องกรอก"], 400);
        }
    }
    
    // อัปเดตข้อมูล
    $result = $personnel->update($id, $input);
    
    if ($result) {
        sendJsonResponse([
            'success' => true,
            'message' => 'อัปเดตข้อมูลบุคลากรสำเร็จ'
        ]);
    } else {
        sendJsonResponse(['error' => 'ไม่สามารถอัปเดตข้อมูลบุคลากรได้'], 500);
    }
}

// DELETE - ลบข้อมูลบุคลากร
function handleDelete($personnel) {
    // ตรวจสอบสิทธิ์
    requireAdmin();
    
    $id = isset($_GET['id']) ? (int)$_GET['id'] : null;
    if (!$id || $id <= 0) {
        sendJsonResponse(['error' => 'ไม่ระบุ ID บุคลากร'], 400);
    }
    
    // ลบข้อมูล (Soft Delete)
    $result = $personnel->delete($id);
    
    if ($result) {
        sendJsonResponse([
            'success' => true,
            'message' => 'ลบข้อมูลบุคลากรสำเร็จ'
        ]);
    } else {
        sendJsonResponse(['error' => 'ไม่สามารถลบข้อมูลบุคลากรได้'], 500);
    }
}
?>
