<?php
/**
 * Authentication API
 * API สำหรับการเข้าสู่ระบบและจัดการผู้ใช้
 */

require_once '../../config/config.php';
require_once '../../models/User.php';

// ตั้งค่า Content-Type
header('Content-Type: application/json; charset=utf-8');

// ตรวจสอบ HTTP Method
$method = $_SERVER['REQUEST_METHOD'];

try {
    $user = new User();
    
    switch ($method) {
        case 'POST':
            handlePost($user);
            break;
        default:
            sendJsonResponse(['error' => 'Method not allowed'], 405);
    }
    
} catch (Exception $e) {
    error_log("Auth API error: " . $e->getMessage());
    sendJsonResponse(['error' => 'เกิดข้อผิดพลาดในระบบ'], 500);
}

// POST - เข้าสู่ระบบ
function handlePost($user) {
    $action = $_GET['action'] ?? 'login';
    
    switch ($action) {
        case 'login':
            handleLogin($user);
            break;
        case 'logout':
            handleLogout();
            break;
        case 'change-password':
            handleChangePassword($user);
            break;
        case 'profile':
            handleProfile($user);
            break;
        default:
            sendJsonResponse(['error' => 'Action not found'], 404);
    }
}

// เข้าสู่ระบบ
function handleLogin($user) {
    // รับข้อมูลจาก Request Body
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || empty($input['username']) || empty($input['password'])) {
        sendJsonResponse(['error' => 'กรุณากรอกชื่อผู้ใช้และรหัสผ่าน'], 400);
    }
    
    $username = sanitizeInput($input['username']);
    $password = $input['password'];
    
    // ตรวจสอบความยาวของข้อมูล
    if (strlen($username) < 3 || strlen($username) > 50) {
        sendJsonResponse(['error' => 'ชื่อผู้ใช้ต้องมี 3-50 ตัวอักษร'], 400);
    }
    
    if (strlen($password) < 6) {
        sendJsonResponse(['error' => 'รหัสผ่านต้องมีอย่างน้อย 6 ตัวอักษร'], 400);
    }
    
    // เข้าสู่ระบบ
    $user_data = $user->login($username, $password);
    
    if ($user_data) {
        // ตั้งค่า Session
        $_SESSION['user_id'] = $user_data['id'];
        $_SESSION['username'] = $user_data['username'];
        $_SESSION['full_name'] = $user_data['full_name'];
        $_SESSION['role'] = $user_data['role'];
        $_SESSION['login_time'] = time();
        
        // บันทึกการเข้าสู่ระบบ
        logActivity($user_data['id'], 'LOGIN', 'เข้าสู่ระบบ');
        
        sendJsonResponse([
            'success' => true,
            'message' => 'เข้าสู่ระบบสำเร็จ',
            'data' => [
                'id' => $user_data['id'],
                'username' => $user_data['username'],
                'full_name' => $user_data['full_name'],
                'role' => $user_data['role'],
                'last_login' => $user_data['last_login']
            ]
        ]);
    } else {
        sendJsonResponse(['error' => 'ชื่อผู้ใช้หรือรหัสผ่านไม่ถูกต้อง'], 401);
    }
}

// ออกจากระบบ
function handleLogout() {
    if (isset($_SESSION['user_id'])) {
        // บันทึกการออกจากระบบ
        logActivity($_SESSION['user_id'], 'LOGOUT', 'ออกจากระบบ');
    }
    
    // ลบ Session
    session_destroy();
    
    sendJsonResponse([
        'success' => true,
        'message' => 'ออกจากระบบสำเร็จ'
    ]);
}

// เปลี่ยนรหัสผ่าน
function handleChangePassword($user) {
    // ตรวจสอบการเข้าสู่ระบบ
    requireLogin();
    
    // รับข้อมูลจาก Request Body
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || empty($input['old_password']) || empty($input['new_password'])) {
        sendJsonResponse(['error' => 'กรุณากรอกรหัสผ่านเก่าและรหัสผ่านใหม่'], 400);
    }
    
    $old_password = $input['old_password'];
    $new_password = $input['new_password'];
    
    // ตรวจสอบความยาวรหัสผ่าน
    if (strlen($new_password) < PASSWORD_MIN_LENGTH) {
        sendJsonResponse(['error' => 'รหัสผ่านต้องมีอย่างน้อย ' . PASSWORD_MIN_LENGTH . ' ตัวอักษร'], 400);
    }
    
    // เปลี่ยนรหัสผ่าน
    $result = $user->changePassword($_SESSION['user_id'], $old_password, $new_password);
    
    if ($result) {
        // บันทึกการเปลี่ยนรหัสผ่าน
        logActivity($_SESSION['user_id'], 'CHANGE_PASSWORD', 'เปลี่ยนรหัสผ่าน');
        
        sendJsonResponse([
            'success' => true,
            'message' => 'เปลี่ยนรหัสผ่านสำเร็จ'
        ]);
    } else {
        sendJsonResponse(['error' => 'รหัสผ่านเก่าไม่ถูกต้อง'], 400);
    }
}

// ดูข้อมูลโปรไฟล์
function handleProfile($user) {
    // ตรวจสอบการเข้าสู่ระบบ
    requireLogin();
    
    $user_data = $user->getById($_SESSION['user_id']);
    
    if ($user_data) {
        sendJsonResponse([
            'success' => true,
            'data' => $user_data
        ]);
    } else {
        sendJsonResponse(['error' => 'ไม่พบข้อมูลผู้ใช้'], 404);
    }
}
?>
